<?php
/*
Plugin Name: Contact Button by BongoIT
Plugin URI: https://bongo.it.com/contact-button
Description: Add a customizable floating contact button with WhatsApp, Messenger, Call, Telegram, and App options. Fully responsive and animated.
Version: 1.0
Author: Mehedi Hassan
Author URI: https://bongo.it.com
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: contact-button-by-bongoit
*/

if ( ! defined( 'ABSPATH' ) ) { exit; }

define( 'CB_BONGOIT_VER', '1.0.0' );
define( 'CB_BONGOIT_URL', plugin_dir_url( __FILE__ ) );
define( 'CB_BONGOIT_PATH', plugin_dir_path( __FILE__ ) );
define( 'CB_BONGOIT_ASSETS_URL', CB_BONGOIT_URL . 'assets/' );
define( 'CB_BONGOIT_ASSETS_PATH', CB_BONGOIT_PATH . 'assets/' );

/** Helpers */
function cb_bongoit_file_ver_abs( $abs_path, $fallback = CB_BONGOIT_VER ) {
    return file_exists( $abs_path ) ? (string) filemtime( $abs_path ) : $fallback;
}

/**
 * Enqueue frontend assets
 * - Font Awesome from ROOT: vendor/fontawesome/css/all.min.css
 * - Your style & js from assets/
 */
function cb_enqueue_frontend_assets() {
    // Font Awesome (ROOT/vendor)
    $fa_css_rel  = 'vendor/fontawesome/css/all.min.css';
    $fa_css_url  = CB_BONGOIT_URL  . $fa_css_rel;
    $fa_css_path = CB_BONGOIT_PATH . $fa_css_rel;

    wp_enqueue_style(
        'cb-fontawesome',
        $fa_css_url,
        [],
        cb_bongoit_file_ver_abs( $fa_css_path )
    );

    // Plugin Style (assets/style.css)
    $style_rel  = 'style.css';
    $style_url  = CB_BONGOIT_ASSETS_URL  . $style_rel;
    $style_path = CB_BONGOIT_ASSETS_PATH . $style_rel;

    wp_enqueue_style(
        'cb-style',
        $style_url,
        array( 'cb-fontawesome' ), // ensure FA loads before
        cb_bongoit_file_ver_abs( $style_path )
    );

    // Plugin Script (assets/front.js)
    $js_rel  = 'front.js';
    $js_url  = CB_BONGOIT_ASSETS_URL  . $js_rel;
    $js_path = CB_BONGOIT_ASSETS_PATH . $js_rel;

    wp_enqueue_script(
        'cb-script',
        $js_url,
        [],
        cb_bongoit_file_ver_abs( $js_path ),
        true
    );
}
add_action( 'wp_enqueue_scripts', 'cb_enqueue_frontend_assets' );

/**
 * Enqueue admin assets
 * - Load FA only on our admin page
 */
function cb_enqueue_admin_assets( $hook ) {
    if ( strpos( $hook, 'cb-contact-button' ) === false ) {
        return;
    }

    $fa_css_rel  = 'vendor/fontawesome/css/all.min.css';
    $fa_css_url  = CB_BONGOIT_URL  . $fa_css_rel;
    $fa_css_path = CB_BONGOIT_PATH . $fa_css_rel;

    wp_enqueue_style(
        'cb-fontawesome',
        $fa_css_url,
        [],
        cb_bongoit_file_ver_abs( $fa_css_path )
    );

    // (ঐচ্ছিক) তোমার অ্যাডমিন স্টাইল থাকলে এখানে enqueue করো
}
add_action( 'admin_enqueue_scripts', 'cb_enqueue_admin_assets' );

/**
 * Settings link on plugin list
 */
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'cb_add_settings_link' );
function cb_add_settings_link( $links ) {
    $url = add_query_arg( [ 'page' => 'cb-contact-button' ], admin_url( 'admin.php' ) );
    $settings_link = '<a href="' . esc_url( $url ) . '">' . esc_html__( 'Settings', 'contact-button-by-bongoit' ) . '</a>';
    array_unshift( $links, $settings_link );
    return $links;
}

/** Admin page include */
require_once plugin_dir_path( __FILE__ ) . 'admin/settings-page.php';

/**
 * Frontend render – floating buttons.
 */
add_action( 'wp_footer', 'cb_render_floating_button' );
function cb_render_floating_button() {
    $opt = get_option( 'cb_contact_button_settings' );
    if ( ! is_array( $opt ) ) { return; }

    $mainText = isset( $opt['main_text'] ) ? sanitize_text_field( $opt['main_text'] ) : '';
    $bg       = isset( $opt['bg_color'] ) ? sanitize_hex_color( $opt['bg_color'] ) : '#e74c3c';
    $color    = isset( $opt['text_icon_color'] ) ? sanitize_hex_color( $opt['text_icon_color'] ) : '#ffffff';

    $enabled = [];
    if ( ! empty( $opt['enable_whatsapp'] )  && ! empty( $opt['whatsapp_number'] ) )   $enabled[] = 'whatsapp';
    if ( ! empty( $opt['enable_messenger'] ) && ! empty( $opt['messenger_username'] ) ) $enabled[] = 'messenger';
    if ( ! empty( $opt['enable_call'] )      && ! empty( $opt['phone_link'] ) )         $enabled[] = 'call';
    if ( ! empty( $opt['enable_telegram'] )  && ! empty( $opt['telegram_link'] ) )      $enabled[] = 'telegram';
    if ( ! empty( $opt['enable_other'] )     && ! empty( $opt['app_link'] ) )           $enabled[] = 'other';

    if ( empty( $enabled ) ) { return; }

    echo '<div id="cb-contact-wrapper">';
    echo '<div id="cb-contact-buttons" class="cb-hidden">';

    // App (Google Play)
    if ( in_array( 'other', $enabled, true ) ) {
        $app_url = isset( $opt['app_link'] ) ? $opt['app_link'] : '';
        echo '<a class="cb-btn app" href="' . esc_url( $app_url ) . '" target="_blank" rel="noopener">
                <i class="fab fa-google-play" aria-hidden="true"></i>
              </a>';
    }

    // Telegram
    if ( in_array( 'telegram', $enabled, true ) ) {
        $tg = isset( $opt['telegram_link'] ) ? $opt['telegram_link'] : '';
        echo '<a class="cb-btn telegram" href="' . esc_url( $tg ) . '" target="_blank" rel="noopener">
                <i class="fab fa-telegram-plane" aria-hidden="true"></i>
              </a>';
    }

    // Call
    if ( in_array( 'call', $enabled, true ) ) {
        $phone = isset( $opt['phone_link'] ) ? preg_replace( '/[^0-9+\-()]/', '', $opt['phone_link'] ) : '';
        echo '<a class="cb-btn call" href="tel:' . esc_attr( $phone ) . '">
                <i class="fas fa-phone" aria-hidden="true"></i>
              </a>';
    }

    // Messenger
    if ( in_array( 'messenger', $enabled, true ) ) {
        $msgr_user = isset( $opt['messenger_username'] ) ? strtolower( preg_replace( '/[^a-z0-9\.]/', '', $opt['messenger_username'] ) ) : '';
        $msgr_url  = 'https://m.me/' . $msgr_user;
        echo '<a class="cb-btn messenger" href="' . esc_url( $msgr_url ) . '" target="_blank" rel="noopener">
                <i class="fab fa-facebook-messenger" aria-hidden="true"></i>
              </a>';
    }

    // WhatsApp
    if ( in_array( 'whatsapp', $enabled, true ) ) {
        $wa_number       = isset( $opt['whatsapp_number'] ) ? preg_replace( '/[^0-9]/', '', $opt['whatsapp_number'] ) : '';
        $is_product      = ( function_exists( 'is_product' ) ? is_product() : is_singular( 'product' ) );
        $product_link    = $is_product ? get_permalink() : '';
        $product_message = isset( $opt['product_page_message'] ) ? sanitize_text_field( $opt['product_page_message'] ) : 'Hello, I want to order this product';
        $default_message = isset( $opt['default_message'] ) ? sanitize_text_field( $opt['default_message'] ) : 'Hello, I need help.';

        $wa_text = $is_product ? trim( $product_message . ' ' . $product_link ) : $default_message;
        $wa_url  = add_query_arg( [ 'text' => $wa_text ], 'https://wa.me/' . $wa_number );

        echo '<a class="cb-btn whatsapp" href="' . esc_url( $wa_url ) . '" target="_blank" rel="noopener">
                <i class="fab fa-whatsapp" aria-hidden="true"></i>
              </a>';
    }

    echo '</div>';

    // Main button
    $style = sprintf( 'background:%1$s; color:%2$s;', esc_attr( $bg ), esc_attr( $color ) );
    echo '<div id="cb-main-btn" class="cb-floating-button" style="' . esc_attr( $style ) . '">';
    echo    '<span class="cb-button-text" style="color:' . esc_attr( $color ) . ';">' . esc_html( $mainText ) . '</span>';
    echo    '<span class="cb-button-icon"><i class="fas fa-headset" aria-hidden="true"></i></span>';
    echo '</div>';

    echo '</div>';
}