<?php
// Admin menu
add_action('admin_menu', 'cb_register_settings_page');

function cb_register_settings_page() {
    add_menu_page(
        'Contact Button',
        'Contact Button',
        'manage_options',
        'cb-contact-button',
        'cb_render_settings_page',
        'dashicons-format-chat',
        80
    );
}

function cb_render_settings_page() {
    $tabs = [
        'main'      => 'Main Button Settings',
        'whatsapp'  => 'WhatsApp Settings',
        'messenger' => 'Messenger Settings',
        'call'      => 'Call Settings',
        'telegram'  => 'Telegram Settings',
        'other'     => 'App Settings',
    ];

    // Safe GET reads (no direct $_GET)
    $allowed_tabs = array_keys($tabs);
    $active_tab   = filter_input(INPUT_GET, 'tab', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $active_tab   = ($active_tab && in_array($active_tab, $allowed_tabs, true)) ? $active_tab : 'main';

    $is_updated = (bool) filter_input(INPUT_GET, 'updated', FILTER_VALIDATE_BOOLEAN);

    $options = get_option('cb_contact_button_settings', []);
    ?>

    <div class="wrap">
        <h1><?php echo esc_html('Contact Button Settings'); ?></h1>

        <h2 class="nav-tab-wrapper">
            <?php foreach ($tabs as $slug => $label) :
                $url = add_query_arg(
                    ['page' => 'cb-contact-button', 'tab' => $slug],
                    admin_url('admin.php')
                );
                ?>
                <a href="<?php echo esc_url($url); ?>"
                   class="nav-tab <?php echo ($active_tab === $slug) ? 'nav-tab-active' : ''; ?>">
                    <?php echo esc_html($label); ?>
                </a>
            <?php endforeach; ?>
        </h2>

        <?php if ($is_updated) : ?>
            <div class="notice notice-success is-dismissible">
                <p><?php echo esc_html__('Settings saved.', 'contact-button-by-bongoit'); ?></p>
            </div>
        <?php endif; ?>

        <form method="post" action="options.php">
            <?php
            settings_fields('cb_settings_group');
            do_settings_sections('cb-contact-button-' . esc_attr($active_tab));
            submit_button(esc_html__('Save Changes', 'contact-button-by-bongoit'));
            ?>
            <input type="hidden" name="cb_current_tab" value="<?php echo esc_attr($active_tab); ?>" />
        </form>

        <?php if ('main' === $active_tab) : ?>
            <h3 style="margin-top: 30px;"><?php echo esc_html__('Live Preview', 'contact-button-by-bongoit'); ?></h3>
            <div id="cb-preview-container" style="margin-top: 10px;">
                <?php
                $bg    = isset($options['bg_color']) ? sanitize_hex_color($options['bg_color']) : '#e74c3c';
                $color = isset($options['text_icon_color']) ? sanitize_hex_color($options['text_icon_color']) : '#ffffff';
                $text  = isset($options['main_text']) ? sanitize_text_field($options['main_text']) : '';

                $style = sprintf(
                    'display:inline-flex;align-items:center;gap:10px;padding:10px 16px;border-radius:50px;background-color:%1$s;color:%2$s;font-weight:600;',
                    esc_attr($bg),
                    esc_attr($color)
                );
                ?>
                <div class="cb-floating-button" style="<?php echo esc_attr($style); ?>">
                    <?php if (!empty($text)) : ?>
                        <span class="cb-button-text"><?php echo esc_html($text); ?></span>
                    <?php endif; ?>
                    <span class="cb-button-icon"><i class="fas fa-headset" aria-hidden="true"></i></span>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function () {
        document.querySelectorAll('.cb-color-circle').forEach(function (btn) {
            btn.addEventListener('click', function () {
                const key = btn.getAttribute('data-target');
                const color = btn.getAttribute('data-color');
                const input = document.querySelector('input[name="cb_contact_button_settings[' + key + ']"]');
                const hexInput = document.querySelector('input.cb-hex-input[data-target="' + key + '"]');
                if (input && hexInput) {
                    input.value = color;
                    hexInput.value = color;
                    input.dispatchEvent(new Event('input'));
                }
            });
        });

        document.querySelectorAll('input.cb-hex-input').forEach(function (input) {
            input.addEventListener('input', function () {
                const key = input.getAttribute('data-target');
                const colorInput = document.querySelector('input[name="cb_contact_button_settings[' + key + ']"]');
                if (colorInput) {
                    colorInput.value = input.value;
                    colorInput.dispatchEvent(new Event('input'));
                }
            });
        });
    });
    </script>

    <style>
    .cb-color-group{display:flex;align-items:center;gap:10px;margin-top:6px;margin-bottom:8px}
    .cb-color-circle{width:24px;height:24px;border-radius:50%;border:2px solid #ccc;cursor:pointer}
    input[type="color"]{width:40px;height:36px;padding:0;border:none;background:none;cursor:pointer}
    input.cb-hex-input{width:140px;height:36px;padding:4px 8px}
    </style>
<?php
}

// Settings & fields
add_action('admin_init', 'cb_register_settings');

function cb_register_settings() {
    register_setting('cb_settings_group', 'cb_contact_button_settings', [
        'sanitize_callback' => 'cb_sanitize_settings',
    ]);

    $fields = [
        'main' => [
            ['main_text', 'Main Button Text'],
            ['text_icon_color', 'Text & Icon Color'],
            ['bg_color', 'Background Color'],
        ],
        'whatsapp' => [
            ['enable_whatsapp', 'Enable WhatsApp Button', 'checkbox'],
            ['whatsapp_number', 'WhatsApp Number (+8801XXXXXXXXX)'],
            ['product_page_message', 'Product Page Message'],
            ['default_message', 'Other Page Message'],
        ],
        'messenger' => [
            ['enable_messenger', 'Enable Messenger Button', 'checkbox'],
            ['messenger_username', 'Messenger Username'],
        ],
        'call' => [
            ['enable_call', 'Enable Call Button', 'checkbox'],
            ['phone_link', 'Phone Number (tel:)'],
        ],
        'telegram' => [
            ['enable_telegram', 'Enable Telegram Button', 'checkbox'],
            ['telegram_link', 'Telegram Link'],
        ],
        'other' => [
            ['enable_other', 'Enable App Button', 'checkbox'],
            ['app_link', 'Google Play App Link'],
        ],
    ];

    foreach ($fields as $tab => $tab_fields) {
        add_settings_section("cb_section_$tab", '', null, "cb-contact-button-$tab");

        foreach ($tab_fields as $field) {
            $key   = $field[0];
            $label = $field[1];
            $type  = isset($field[2]) ? $field[2] : (strpos($key, 'color') !== false ? 'color' : 'text');

            add_settings_field(
                $key,
                esc_html($label),
                function () use ($key, $type) {
                    $opt = get_option('cb_contact_button_settings', []);
                    $defaults = [
                        'product_page_message' => 'Hello, I want order this product',
                        'default_message'      => 'Hello, I need some help.',
                    ];
                    $val = isset($opt[$key]) ? $opt[$key] : (isset($defaults[$key]) ? $defaults[$key] : '');

                    if ('checkbox' === $type) {
                        printf('<input type="hidden" name="cb_contact_button_settings[%1$s]" value="0" />', esc_attr($key));
                        printf(
                            '<label><input type="checkbox" name="cb_contact_button_settings[%1$s]" value="1"%2$s /> %3$s</label>',
                            esc_attr($key),
                            checked('1', $val, false),
                            esc_html__('Enable', 'contact-button-by-bongoit')
                        );
                    } elseif ('color' === $type) {
                        printf(
                            '<div class="cb-color-group"><input type="color" name="cb_contact_button_settings[%1$s]" value="%2$s" />',
                            esc_attr($key),
                            esc_attr($val)
                        );
                        printf(
                            '<input class="cb-hex-input" data-target="%1$s" type="text" value="%2$s" /></div>',
                            esc_attr($key),
                            esc_attr($val)
                        );
                        echo '<div class="cb-color-group">';
                        $colors = ['#ffffff', '#000000', '#16A34A', '#F97316', '#1E3A8A'];
                        foreach ($colors as $swatch) {
                            printf(
                                '<div class="cb-color-circle" style="background:%1$s" data-target="%2$s" data-color="%1$s"></div>',
                                esc_attr($swatch),
                                esc_attr($key)
                            );
                        }
                        echo '</div>';
                    } else {
                        printf(
                            '<input type="%1$s" name="cb_contact_button_settings[%2$s]" value="%3$s" class="regular-text" />',
                            esc_attr($type),
                            esc_attr($key),
                            esc_attr($val)
                        );
                    }
                },
                "cb-contact-button-$tab",
                "cb_section_$tab"
            );
        }
    }
}

// Sanitizer
function cb_sanitize_settings($input) {
    $sanitized = [];

    if (!is_array($input)) {
        return $sanitized;
    }

    foreach ($input as $key => $value) {
        if (strpos($key, 'enable_') === 0) {
            $sanitized[$key] = ($value === '1') ? '1' : '0';
        } elseif (strpos($key, 'color') !== false) {
            $sanitized[$key] = sanitize_hex_color($value);
        } elseif (strpos($key, 'link') !== false || strpos($key, 'url') !== false) {
            $sanitized[$key] = esc_url_raw($value);
        } else {
            $sanitized[$key] = sanitize_text_field($value);
        }
    }
    return $sanitized;
}

// Secure handler for form submission (no direct $_POST)
add_action('admin_init', function () {
    $post = filter_input_array(INPUT_POST, [
        'option_page'                => FILTER_DEFAULT,
        '_wpnonce'                   => FILTER_DEFAULT,
        'cb_contact_button_settings' => [
            'filter' => FILTER_DEFAULT,
            'flags'  => FILTER_REQUIRE_ARRAY,
        ],
        'cb_current_tab'             => FILTER_SANITIZE_FULL_SPECIAL_CHARS,
    ]);

    if (empty($post) || !is_array($post)) {
        return;
    }
    if (!isset($post['option_page']) || 'cb_settings_group' !== $post['option_page']) {
        return;
    }
    if (!current_user_can('manage_options')) {
        return;
    }
    // Matches settings_fields('cb_settings_group') → action 'cb_settings_group-options'
    if (empty($post['_wpnonce']) || !check_admin_referer('cb_settings_group-options')) {
        return;
    }

    $submitted = !empty($post['cb_contact_button_settings']) && is_array($post['cb_contact_button_settings'])
        ? cb_sanitize_settings(wp_unslash($post['cb_contact_button_settings']))
        : [];

    $existing = get_option('cb_contact_button_settings', []);
    $merged   = array_merge($existing, $submitted);
    update_option('cb_contact_button_settings', $merged);

    $tab = !empty($post['cb_current_tab']) ? sanitize_text_field($post['cb_current_tab']) : 'main';

    $redirect_url = add_query_arg(
        ['page' => 'cb-contact-button', 'tab' => $tab, 'updated' => 'true'],
        admin_url('admin.php')
    );
    wp_safe_redirect($redirect_url);
    exit;
});