<?php
/**
 * Secure settings handler
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * We keep this file for handling a custom save action (if you use it).
 * NOTE: If you already save via options.php + register_setting, you may
 * not need this file. But if you post to this handler, it's now safe.
 */

// Server method (no direct $_SERVER)
$method = filter_input( INPUT_SERVER, 'REQUEST_METHOD', FILTER_SANITIZE_FULL_SPECIAL_CHARS );

if ( 'POST' === $method ) {

	// Read a minimal set of expected POST vars safely
	$post = filter_input_array(
		INPUT_POST,
		[
			'cb_save_settings'   => FILTER_DEFAULT, // presence check
			'_wpnonce'           => FILTER_DEFAULT, // for check_admin_referer
			'main_text'          => FILTER_DEFAULT,
			'bg_color'           => FILTER_DEFAULT,
			'text_icon_color'    => FILTER_DEFAULT,
			'enabled_buttons'    => [
				'filter' => FILTER_DEFAULT,
				'flags'  => FILTER_REQUIRE_ARRAY,
			],
			'whatsapp_number'    => FILTER_DEFAULT,
			'messenger_username' => FILTER_DEFAULT,
			'phone_link'         => FILTER_DEFAULT,
			'telegram_link'      => FILTER_DEFAULT,
			'app_link'           => FILTER_DEFAULT,
		]
	);

	// Bail if this is not our form
	if ( empty( $post['cb_save_settings'] ) ) {
		return;
	}

	// Permission
	if ( ! current_user_can( 'manage_options' ) ) {
		return;
	}

	// Nonce (your form should print wp_nonce_field( 'cb_save_settings_verify' ))
	if ( empty( $post['_wpnonce'] ) || ! check_admin_referer( 'cb_save_settings_verify' ) ) {
		return;
	}

	// Sanitize all fields (use wp_unslash() before sanitize)
	$data = [
		'main_text'          => sanitize_text_field( wp_unslash( $post['main_text'] ?? '' ) ),
		'bg_color'           => sanitize_hex_color( wp_unslash( $post['bg_color'] ?? '#e74c3c' ) ),
		'text_icon_color'    => sanitize_hex_color( wp_unslash( $post['text_icon_color'] ?? '#ffffff' ) ),
		'enabled_buttons'    => '',
		'whatsapp_number'    => preg_replace( '/[^0-9]/', '', wp_unslash( $post['whatsapp_number'] ?? '' ) ),
		'messenger_username' => strtolower( preg_replace( '/[^a-z0-9\.]/', '', wp_unslash( $post['messenger_username'] ?? '' ) ) ),
		'phone_link'         => preg_replace( '/[^0-9+\-\(\)\s]/', '', wp_unslash( $post['phone_link'] ?? '' ) ),
		'telegram_link'      => esc_url_raw( wp_unslash( $post['telegram_link'] ?? '' ) ),
		'app_link'           => esc_url_raw( wp_unslash( $post['app_link'] ?? '' ) ),
	];

	// enabled buttons (array → comma list)
	if ( ! empty( $post['enabled_buttons'] ) && is_array( $post['enabled_buttons'] ) ) {
		$clean = array_map(
			static function ( $v ) {
				return sanitize_key( wp_unslash( $v ) );
			},
			$post['enabled_buttons']
		);
		$data['enabled_buttons'] = implode( ',', array_unique( $clean ) );
	}

	// Save
	update_option( 'cb_contact_button_settings', $data );

	// Redirect back with notice (no direct header())
	$redirect_url = add_query_arg(
		[
			'page'   => 'cb-contact-settings', // keep this slug same as your menu page
			'saved'  => 'true',
		],
		admin_url( 'admin.php' )
	);
	wp_safe_redirect( $redirect_url );
	exit;
}